<?php

declare(strict_types=1);

namespace User\Application\CommandHandlers;

use Affiliate\Domain\Exceptions\AffiliateNotFoundException;
use Affiliate\Domain\Repositories\AffiliateRepositoryInterface;
use Psr\EventDispatcher\EventDispatcherInterface;
use User\Application\Commands\CreateUserCommand;
use User\Domain\Entities\UserEntity;
use User\Domain\Events\UserCreatedEvent;
use User\Domain\Exceptions\EmailTakenException;
use User\Domain\Repositories\UserRepositoryInterface;
use User\Infrastructure\Services\EmailChecker;

class CreateUserCommandHandler
{
    public function __construct(
        private UserRepositoryInterface        $repo,
        private AffiliateRepositoryInterface   $affiliateRepo,
        private EventDispatcherInterface       $dispatcher,
        private EmailChecker                   $service
    ) {}

    /**
     * @throws EmailTakenException
     */
    public function handle(CreateUserCommand $cmd): UserEntity
    {
        /* -------- E-mail já usado? -------- */
        if (!$this->service->isValid($cmd->email)) {
            throw new EmailTakenException($cmd->email);
        }

        /* -------- Instância do usuário -------- */
        $user = new UserEntity(
            email     : $cmd->email,
            firstName : $cmd->firstName,
            lastName  : $cmd->lastName
        );

        /* -------- Campos opcionais -------- */
        if ($cmd->password)      { $user->setPassword($cmd->password); }
        if ($cmd->phoneNumber)   { $user->setPhoneNumber($cmd->phoneNumber); }
        if ($cmd->language)      { $user->setLanguage($cmd->language); }
        if ($cmd->ip)            { $user->setIp($cmd->ip); }
        if ($cmd->countryCode)   { $user->setCountryCode($cmd->countryCode); }
        if ($cmd->cityName)      { $user->setCityName($cmd->cityName); }
        if ($cmd->role)          { $user->setRole($cmd->role); }
        if ($cmd->status)        { $user->setStatus($cmd->status); }
        if ($cmd->workspaceCap)  { $user->setWorkspaceCap($cmd->workspaceCap); }

        /* -------- Novos campos -------- */
        if ($cmd->cpf  !== null) { $user->setCpf($cmd->cpf); }
        if ($cmd->crm  !== null) { $user->setCrm($cmd->crm); }
        if ($cmd->estado !== null)      { $user->setEstado($cmd->estado); }
        if ($cmd->matricula !== null)   { $user->setMatricula($cmd->matricula); }
        if ($cmd->instituicao !== null) { $user->setInstituicao($cmd->instituicao); }

        /* -------- Afiliado (ref code) -------- */
        if ($cmd->refCode) {
            try {
                $affiliate = $this->affiliateRepo->ofCode($cmd->refCode);
                $user->setReferredBy($affiliate->getUser());
            } catch (AffiliateNotFoundException) {
                // Ignora ref codes inválidos
            }
        }

        /* -------- Confirmação de e-mail (opcional) -------- */
        if ($cmd->isEmailVerified !== null && $cmd->isEmailVerified->value) {
            $user->verifyEmail($user->getEmailVerificationToken());
        }

        /* -------- Persiste e dispara evento -------- */
        $this->repo->add($user);
        $this->dispatcher->dispatch(new UserCreatedEvent($user));

        return $user;
    }
}
