<?php

declare(strict_types=1);

namespace User\Domain\Entities;

use Affiliate\Domain\Entities\AffiliateEntity;
use DateTime;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Exception;
use Ramsey\Uuid\Uuid;
use Shared\Domain\ValueObjects\CityName;
use Shared\Domain\ValueObjects\CountryCode;
use Shared\Domain\ValueObjects\Id;
use Shared\Domain\ValueObjects\Ip;
use Traversable;
use User\Domain\Exceptions\InvalidPasswordException;
use User\Domain\Exceptions\InvalidTokenException;
use User\Domain\Exceptions\OwnedWorkspaceCapException;
use User\Domain\ValueObjects\ApiKey;
use User\Domain\ValueObjects\Email;
use User\Domain\ValueObjects\EmailVerificationToken;
use User\Domain\ValueObjects\FirstName;
use User\Domain\ValueObjects\IsEmailVerified;
use User\Domain\ValueObjects\Language;
use User\Domain\ValueObjects\LastName;
use User\Domain\ValueObjects\Password;
use User\Domain\ValueObjects\PasswordHash;
use User\Domain\ValueObjects\PhoneNumber;
use User\Domain\ValueObjects\Preferences;
use User\Domain\ValueObjects\RecoveryToken;
use User\Domain\ValueObjects\Role;
use User\Domain\ValueObjects\Status;
use User\Domain\ValueObjects\WorkspaceCap;
use Workspace\Domain\Entities\WorkspaceEntity;
use Workspace\Domain\Exceptions\WorkspaceNotFoundException;
use Workspace\Domain\ValueObjects\Name;

#[ORM\Entity]
#[ORM\Table(name: 'user')]
#[ORM\Index(columns: ['first_name'])]
#[ORM\Index(columns: ['last_name'])]
#[ORM\HasLifecycleCallbacks]
class UserEntity
{
    /** Number of seconds after which a user is considered offline */
    public const ONLINE_THRESHOLD = 300; // 5 minutes

    #[ORM\Embedded(class: Id::class, columnPrefix: false)]
    private Id $id;

    #[ORM\Column(type: Types::SMALLINT, enumType: Role::class, name: 'role')]
    private Role $role;

    #[ORM\Embedded(class: Email::class, columnPrefix: false)]
    private Email $email;

    #[ORM\Embedded(class: PasswordHash::class, columnPrefix: false)]
    private PasswordHash $passwordHash;

    #[ORM\Embedded(class: FirstName::class, columnPrefix: false)]
    private FirstName $firstName;

    #[ORM\Embedded(class: LastName::class, columnPrefix: false)]
    private LastName $lastName;

    #[ORM\Embedded(class: PhoneNumber::class, columnPrefix: false)]
    private PhoneNumber $phoneNumber;

    #[ORM\Column(type: Types::STRING, name: 'cpf', length: 14, nullable: true)]
    private ?string $cpf = null;

    #[ORM\Column(type: Types::STRING, name: 'crm', length: 20, nullable: true)]
    private ?string $crm = null;

    #[ORM\Column(type: Types::STRING, name: 'estado', length: 2, nullable: true)]
    private ?string $estado = null;

    #[ORM\Column(type: Types::STRING, name: 'matricula', length: 50, nullable: true)]
    private ?string $matricula = null;

    #[ORM\Column(type: Types::STRING, name: 'instituicao', length: 255, nullable: true)]
    private ?string $instituicao = null;

    #[ORM\Embedded(class: Language::class, columnPrefix: false)]
    private Language $language;

    #[ORM\Embedded(class: ApiKey::class, columnPrefix: false)]
    private ApiKey $apiKey;

    #[ORM\Embedded(class: Ip::class, columnPrefix: false)]
    private Ip $ip;

    #[ORM\Column(type: Types::STRING, enumType: CountryCode::class, name: 'country_code', nullable: true)]
    private ?CountryCode $countryCode = null;

    #[ORM\Embedded(class: CityName::class, columnPrefix: false)]
    private CityName $cityName;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, name: 'created_at')]
    private DateTimeInterface $createdAt;

    #[ORM\Column(type: Types::DATETIME_MUTABLE, name: 'updated_at', nullable: true)]
    private ?DateTimeInterface $updatedAt = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE, name: 'last_seen_at', nullable: true)]
    private ?DateTimeInterface $lastSeenAt = null;

    #[ORM\Column(type: Types::SMALLINT, enumType: Status::class, name: 'status')]
    private Status $status;

    #[ORM\Embedded(class: RecoveryToken::class, columnPrefix: false)]
    private RecoveryToken $recoveryToken;

    #[ORM\Embedded(class: IsEmailVerified::class, columnPrefix: false)]
    private IsEmailVerified $isEmailVerified;

    #[ORM\Embedded(class: EmailVerificationToken::class, columnPrefix: false)]
    private EmailVerificationToken $emailVerificationToken;

    #[ORM\Embedded(class: WorkspaceCap::class, columnPrefix: false)]
    private WorkspaceCap $workspaceCap;

    /** @var Collection<int,WorkspaceEntity> */
    #[ORM\ManyToMany(targetEntity: WorkspaceEntity::class, mappedBy: 'users')]
    private Collection $workspaces;

    /** @var Collection<int,WorkspaceEntity> */
    #[ORM\OneToMany(targetEntity: WorkspaceEntity::class, mappedBy: 'owner', cascade: ['persist', 'remove'])]
    private Collection $ownedWorkspaces;

    #[ORM\ManyToOne(targetEntity: WorkspaceEntity::class)]
    #[ORM\JoinColumn(name: 'current_workspace_id', nullable: true, onDelete: 'SET NULL')]
    private ?WorkspaceEntity $currentWorkspace = null;

    #[ORM\OneToOne(targetEntity: AffiliateEntity::class, cascade: ['persist', 'remove'], mappedBy: 'user')]
    private ?AffiliateEntity $affiliate = null;

    #[ORM\ManyToOne(targetEntity: UserEntity::class)]
    #[ORM\JoinColumn(name: 'referred_by', nullable: true, onDelete: 'SET NULL')]
    private ?UserEntity $referredBy = null;

    #[ORM\Column(type: Types::JSON, name: 'preferences', nullable: true)]
    private null|array|Preferences $preferences = null;

    public function __construct(Email $email, FirstName $firstName, LastName $lastName)
    {
        $this->id                  = new Id();
        $this->role                = Role::USER;
        $this->email               = $email;
        $this->passwordHash        = new PasswordHash();
        $this->firstName           = $firstName;
        $this->lastName            = $lastName;
        $this->phoneNumber         = new PhoneNumber();
        $this->language            = new Language();
        $this->apiKey              = new ApiKey();
        $this->ip                  = new Ip();
        $this->cityName            = new CityName();
        $this->createdAt           = new DateTimeImmutable();
        $this->status              = Status::ACTIVE;
        $this->recoveryToken       = new RecoveryToken();
        $this->isEmailVerified     = new IsEmailVerified();
        $this->emailVerificationToken = new EmailVerificationToken(Uuid::uuid4()->toString());
        $this->workspaceCap        = new WorkspaceCap(0);
        $this->workspaces          = new ArrayCollection();
        $this->ownedWorkspaces     = new ArrayCollection();

        $this->createDefaultWorkspace();
        $this->createAffiliate();
    }

    /* Basic Getters / Setters */

    public function getId(): Id { return $this->id; }

    public function getRole(): Role { return $this->role; }
    public function setRole(Role $role): self { $this->role = $role; return $this; }

    public function getEmail(): Email { return $this->email; }

    public function getFirstName(): FirstName { return $this->firstName; }
    public function setFirstName(FirstName $first): self { $this->firstName = $first; return $this; }

    public function getLastName(): LastName { return $this->lastName; }
    public function setLastName(LastName $last): self { $this->lastName = $last; return $this; }

    public function getPhoneNumber(): PhoneNumber { return $this->phoneNumber; }
    public function setPhoneNumber(PhoneNumber $phone): self { $this->phoneNumber = $phone; return $this; }

    /* Extra fields */

    public function getCpf(): ?string { return $this->cpf; }
    public function setCpf(?string $cpf): self { $this->cpf = $cpf; return $this; }

    public function getCrm(): ?string { return $this->crm; }
    public function setCrm(?string $crm): self { $this->crm = $crm; return $this; }

    public function getEstado(): ?string { return $this->estado; }
    public function setEstado(?string $estado): self { $this->estado = $estado; return $this; }

    public function getMatricula(): ?string { return $this->matricula; }
    public function setMatricula(?string $matricula): self { $this->matricula = $matricula; return $this; }

    public function getInstituicao(): ?string { return $this->instituicao; }
    public function setInstituicao(?string $instituicao): self { $this->instituicao = $instituicao; return $this; }

    /* Remaining getters / setters */

    public function getLanguage(): Language { return $this->language; }
    public function setLanguage(Language $lang): self { $this->language = $lang; return $this; }

    public function getApiKey(): ApiKey { return $this->apiKey; }
    public function generateApiKey(): void { $this->apiKey = new ApiKey(); }

    public function getIp(): Ip { return $this->ip; }
    public function setIp(Ip $ip): self { $this->ip = $ip; return $this; }

    public function getCountryCode(): ?CountryCode { return $this->countryCode; }
    public function setCountryCode(?CountryCode $cc): self { $this->countryCode = $cc; return $this; }

    public function getCityName(): CityName { return $this->cityName; }
    public function setCityName(CityName $city): self { $this->cityName = $city; return $this; }

    public function getCreatedAt(): DateTimeInterface { return $this->createdAt; }
    public function getUpdatedAt(): ?DateTimeInterface { return $this->updatedAt; }
    public function getLastSeenAt(): ?DateTimeInterface { return $this->lastSeenAt; }

    public function getPreferences(): Preferences
    {
        if (!$this->preferences instanceof Preferences) {
            $this->preferences = new Preferences($this->preferences);
        }
        return $this->preferences;
    }
    public function setPreferences(array|Preferences $pref): self
    {
        if (is_array($pref)) { $pref = new Preferences($pref); }
        $this->preferences = $this->getPreferences()->mergeWith($pref);
        return $this;
    }

    /* Status / online */

    public function isOnline(): bool
    {
        return $this->status === Status::ACTIVE
            && $this->lastSeenAt
            && $this->lastSeenAt->getTimestamp() >= time() - self::ONLINE_THRESHOLD;
    }
    public function touch(): self { $this->lastSeenAt = new DateTime(); return $this; }

    public function getStatus(): Status { return $this->status; }
    public function setStatus(Status $status): self
    {
        if ($status === Status::ONLINE || $status === Status::AWAY) { $status = Status::ACTIVE; }
        $this->status = $status;
        return $this;
    }

    /* Token & email */

    public function getRecoveryToken(): RecoveryToken { return $this->recoveryToken; }
    public function generateRecoveryToken(): void { $this->recoveryToken = new RecoveryToken(Uuid::uuid4()->toString()); }

    public function validateRecoveryToken(RecoveryToken $token): true
    {
        if ($this->recoveryToken->value !== $token->value) { throw new InvalidTokenException($this, $token); }
        return true;
    }

    public function isEmailVerified(): IsEmailVerified { return $this->isEmailVerified; }

    public function verifyEmail(EmailVerificationToken $token): void
    {
        if ($this->emailVerificationToken->value !== $token->value) { throw new InvalidTokenException($this, $token); }
        $this->isEmailVerified = new IsEmailVerified(true);
        $this->emailVerificationToken = new EmailVerificationToken();
    }

    public function getEmailVerificationToken(): EmailVerificationToken { return $this->emailVerificationToken; }

    public function unverifyEmail(): void
    {
        $this->isEmailVerified = new IsEmailVerified(false);
        $this->emailVerificationToken = new EmailVerificationToken(Uuid::uuid4()->toString());
    }

    /* Workspace cap */

    public function getWorkspaceCap(): WorkspaceCap { return $this->workspaceCap; }
    public function setWorkspaceCap(WorkspaceCap $cap): self { $this->workspaceCap = $cap; return $this; }

    /* Email & password updates */

    public function updateEmail(Email $email, Password $pwd): self
    {
        $this->verifyPassword($pwd);
        $this->email = $email;
        $this->isEmailVerified = new IsEmailVerified(false);
        $this->emailVerificationToken = new EmailVerificationToken(Uuid::uuid4()->toString());
        return $this;
    }

    public function updatePassword(Password $current, Password $new): self
    {
        $this->verifyPassword($current);
        if ($current->value === $new->value) {
            throw new InvalidPasswordException($this, $new, InvalidPasswordException::TYPE_SAME_AS_OLD);
        }
        $this->setPassword($new);
        return $this;
    }

    public function resetPassword(RecoveryToken $token, Password $pwd): self
    {
        $this->validateRecoveryToken($token);
        $this->setPassword($pwd);
        $this->recoveryToken = new RecoveryToken();
        return $this;
    }

    public function verifyPassword(Password $pwd): bool
    {
        if (!$this->passwordHash->value || !password_verify($pwd->value, $this->passwordHash->value)) {
            throw new InvalidPasswordException($this, $pwd, InvalidPasswordException::TYPE_INCORRECT);
        }
        return true;
    }

    public function hasPassword(): bool { return $this->passwordHash->value !== null; }

    public function setPassword(Password $pwd): void
    {
        $this->passwordHash = new PasswordHash(
            $pwd->value ? password_hash($pwd->value, PASSWORD_DEFAULT) : null
        );
    }

    #[ORM\PreUpdate] public function preUpdate(): void { $this->updatedAt = new DateTime(); }

    /* Workspaces */

    public function createWorkspace(Name $name): WorkspaceEntity
    {
        if ($this->workspaceCap->value > 0 && $this->ownedWorkspaces->count() >= $this->workspaceCap->value) {
            throw new OwnedWorkspaceCapException();
        }
        $workspace = new WorkspaceEntity($this, $name);
        $this->ownedWorkspaces->add($workspace);
        $this->currentWorkspace = $workspace;
        return $workspace;
    }

    /** @return Traversable<WorkspaceEntity> */
    public function getWorkspaces(): Traversable { return $this->workspaces->getIterator(); }

    /** @return Traversable<WorkspaceEntity> */
    public function getOwnedWorkspaces(): Traversable { return $this->ownedWorkspaces->getIterator(); }

    #[ORM\PostLoad]
    public function postLoad(): void
    {
        if ($this->ownedWorkspaces->count() === 0) { $this->createDefaultWorkspace(); }
        if (!$this->currentWorkspace)       { $this->currentWorkspace = $this->ownedWorkspaces->first(); }
        if (!$this->affiliate)              { $this->createAffiliate(); }
    }

    private function createDefaultWorkspace(): void { $this->createWorkspace(new Name('Pessoal')); }
    private function createAffiliate(): void       { $this->affiliate = new AffiliateEntity($this); }

    /* Misc */

    public function getAffiliate(): AffiliateEntity { return $this->affiliate; }

    public function getReferredBy(): ?UserEntity { return $this->referredBy; }
    public function setReferredBy(UserEntity $u): void { $this->referredBy = $u; }

    public function getCurrentWorkspace(): WorkspaceEntity { return $this->currentWorkspace; }
    public function setCurrentWorkspace(WorkspaceEntity $w): void { $this->currentWorkspace = $w; }

    public function getWorkspaceById(Id $id): WorkspaceEntity
    {
        foreach ($this->workspaces as $ws) {
            if ($ws->getId()->getValue() === $id->getValue()) { return $ws; }
        }
        foreach ($this->ownedWorkspaces as $ws) {
            if ($ws->getId()->getValue() === $id->getValue()) { return $ws; }
        }
        throw new WorkspaceNotFoundException($id);
    }
}
